/*! \file tlk_sclk_api.h 
**********************************************************************************
*Title:                         Discretix Secured Clock Toolkit header file
*
* Filename:                     tlk_sclk_api.h 
*
* Project, Target, subsystem:   Toolkit, Secured Clock
* 
* Created:                      10.09.2008
*
* Modified:                     10.09.2008
*
* \Author                       Ira Boguslavsky
*
* \Remarks
*           Copyright (C) 2008 by Discretix Technologies Ltd. All Rights reserved.
**********************************************************************************/
#ifndef _TLK_SCLK_API_H_
#define _TLK_SCLK_API_H_

        /*----------- Doxygen Main Page text -----------------------------------------*/
        /*! \mainpage Secured Clock Toolkit API
        </P>
        <h1>Secured Clock Toolkit API Documentation</h1>
        <p>
        <img width="268" height="148" src="DxLogo.jpg">
        </p>
        <P class="ConfidentialNotice" style="TEXT-ALIGN: justify">
        This document may be used 
        in its complete form only and is solely for the use of Discretix employees and 
        authorized Discretix channels or customers. The material herein is proprietary 
        to Discretix Ltd., any unauthorized reproduction of any part thereof is 
        strictly prohibited. The contents of this document are believed to be accurate 
        at the time of distribution. Discretix reserves the right to alter this 
        information at any time without notice. The primary distribution media of this 
        document is soft copy.
        </P>

        **/
#ifdef __cplusplus
extern "C" {
#endif

#include "DX_VOS_BaseTypes.h"
#include "tlk_sclk_api_types.h"
/*TLK_SCLK_Init*/
/**
* @brief     The purpose of this function is to initialize the new master clock descriptor. 
*            It reads the mater clock descriptor data from specific RAM memory.
*            If it was not expected, return an error.
*
* @param workspace_ptr              [in]     - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes       [in]     - Length of the workspace in bytes. The workspace size user can get by calling TLK_SCLK_WorkspaceSizeGet() API.
*
* @return TLK_SCLC_RC_OK                                     - The operation completed successfully.
* @return TLK_SCLK_RC_ERROR_API_NVS_CONFIGURATION            - The NVS configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_RTC_CONFIGURATION            - The RTC configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_TOOLKIT_ALREADY_INITIATED    - The toolkit initialization is done already. 
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NULL_PTR           - NULL Pinter workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NOT_ALIGN          - Unaligned workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE                    - Insufficient workspace given by user
* @return TLK_SCLK_RC_ERROR_API_COOKIE_GENERATE_FAIL         - Problem in cookie generation
* @return TLK_SCLK_RC_ERROR_MC_MASTER_CLK_INTEGRITY          - Master clock integrity compromised
* @return TLK_SCLK_RC_ERROR_MC_SECRET_KEY_CREATE_FAIL        - Problem in secrete key creation
* @return TLK_SCLK_RC_ERROR_MC_MAC_CALCULATE_FAIL            - MAC calculation of service clock failed
* @return TLK_SCLK_RC_ERROR_NVS_INIT_SESSION_FAIL            - In case of SST supports, Problem in session creation
* @return TLK_SCLK_RC_ERROR_NVS_INIT_TRANSACTION_FAIL        - In case of SST supports, Problem in transaction starting
* @return TLK_SCLK_RC_ERROR_NVS_INIT_FILE_OPEN_FAIL          - Problem in NVS file or external file opening
* @return TLK_SCLK_RC_ERROR_NVS_READ_MC_DESC_FAIL            - Cannot read master clock descriptor from VNVS
* @return TLK_SCLK_RC_ERROR_NVS_DECRYPT_MC_DESC_FAIL         - Cannot decrypt master clock descriptor
* @return TLK_SCLK_RC_ERROR_NVS_WRITE_MC_DESC_FAIL           - Cannot write master clock descriptor to VNVS
* @return TLK_SCLK_RC_ERROR_NVS_ENCRYPT_MC_DESC_FAIL         - Cannot encrypt master clock descriptor
**/
DxError_t 
TLK_SCLK_Init(                              DxUint8_t                           *workspace_ptr,
                                            DxUint32_t                           workspaceSizeInBytes);
/*TLK_SCLK_ServiceClkGen*/
/**
* @brief     The purpose of this function is to generate a new descriptor for secured service clock.
*
* @param serviceDescr_ptr            [in/out] - Pointer to the service clock descriptor to be generated.
* @param appId_ptr                   [in]     - String which is unique for the application. This parameter can be undefined (DX_NULL) and it is depends on user.
* @param maxAllowedDriftIn0_1Seconds [in]     - Maximum allowed drift for this clock (in units of 0.1 seconds).
* @param maximumRevalidationPeriod   [in]     - Maximum revalidation period ("grace period"): once the clock is at (validity limit + revalidation period), it reverts to an "unset" state.
* @param readPermission              [in]     - Global read permission: allows applications other than the owner to read the secure service clock.
* @param workspace_ptr               [in]     - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes        [in]     - Length of the workspace in bytes. The workspace size user can get by calling TLK_SCLK_WorkspaceSizeGet() API.
*
* @return TLK_SCLC_RC_OK                                     - The operation completed successfully.
* @return TLK_SCLK_RC_ERROR_API_NVS_CONFIGURATION            - The NVS configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_RTC_CONFIGURATION            - The RTC configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_TOOLKIT_ALREADY_INITIATED    - The toolkit initialization is done already. 
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NULL_PTR           - NULL Pinter workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NOT_ALIGN          - Unaligned workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE                    - Insufficient workspace given by user
* @return TLK_SCLK_RC_ERROR_API_NULL_POINTER                 - Null pointer received a input
* @return TLK_SCLK_RC_ERROR_MC_MAC_CALCULATE_FAIL            - MAC calculation of service clock failed
* @return TLK_SCLK_RC_ERROR_NVS_WRITE_MC_DESC_FAIL           - Cannot write master clock descriptor to VNVS
* @return TLK_SCLK_RC_ERROR_NVS_ENCRYPT_MC_DESC_FAIL         - Cannot encrypt master clock descriptor
* @return TLK_SCLK_RC_ERROR_API_MAC_CALCULATE_FAIL           - MAC calculation failed (Integrity problem)
**/
DxError_t 
TLK_SCLK_ServiceClkGen(                     TLK_SCLK_ServiceClockDescriptor_t   *serviceDescr_ptr,
                                            TLK_SCLK_ApplicationId_t            *appId_ptr,
                                            DxTime_t                             maxAllowedDriftIn0_1Seconds,
                                            DxTime_t                             maximumRevalidationPeriod,
                                            TLK_SCLK_ReadPermission_t            readPermission,
											DxUint8_t                           *serviceClkPwd,
											DxUint32_t                          serviceClkPwdLen,
                                            DxUint8_t                           *workspace_ptr,
                                            DxUint32_t                           workspaceSizeInBytes);


/*TLK_SCLK_ServiceClkRead*/
/**
* @brief     The purpose of this function is to verify and read the service secured clock descriptor. 
*            In case of verification fail, the function shall return the status which may be "valid", 
*            "unset/invalid", or "must revalidates" 
*
* @param serviceDescr_ptr           [in/out] - Pointer to the service clock descriptor to be verified.
* @param appId_ptr                  [in]     - Pointer to the application ID.
* @param currTime_ptr               [out]    - Pointer to the current time.
* @param timeFormat                 [in]     - Format in which time can be represented
* @param status_ptr                 [out]    - Read status ("valid", "unset/invalid", or "must revalidate").
* @param workspace_ptr              [in]     - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes       [in]     - Length of the workspace in bytes. The workspace size user can get by calling TLK_SCLK_WorkspaceSizeGet() API.
*
* @return TLK_SCLC_RC_OK                                     - The operation completed successfully.
* @return TLK_SCLK_RC_ERROR_API_NVS_CONFIGURATION            - The NVS configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_RTC_CONFIGURATION            - The RTC configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_TOOLKIT_ALREADY_INITIATED    - The toolkit initialization is done already. 
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NULL_PTR           - NULL Pinter workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NOT_ALIGN          - Unaligned workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE                    - Insufficient workspace given by user
* @return TLK_SCLK_RC_ERROR_API_NULL_POINTER                 - Null pointer received a input
* @return TLK_SCLK_RC_ERROR_API_INVALID_TIME_FORMAT          - User passed invalid format parameter value (Should be TLK_SCLK_SECONDS_SINCE_FORMAT or TLK_SCLK_DATE_TIME_FORMAT)
* @return TLK_SCLK_RC_ERROR_API_APP_ID_INVALID               - Application Id given by user is differ than application id in descriptor
* @return TLK_SCLK_RC_ERROR_API_COOKIE_FAIL                  - Cookie validation failed (the master clock descriptor and service clock descriptor cookies are different)
* @return TLK_SCLK_RC_ERROR_MC_MAC_CALCULATE_FAIL            - MAC calculation of service clock failed
* @return TLK_SCLK_RC_ERROR_NVS_WRITE_MC_DESC_FAIL           - Cannot write master clock descriptor to VNVS
* @return TLK_SCLK_RC_ERROR_NVS_ENCRYPT_MC_DESC_FAIL         - Cannot encrypt master clock descriptor
* @return TLK_SCLK_RC_ERROR_RTC_TIME_GET                     - Error in RTC time getting
* @return TLK_SCLK_RC_ERROR_RTC_TIME_CONVERT_FAIL            - Problem in converting the given time Generalized time to number of seconds since January 1, 1970. or conversion time that is represented as the number of seconds since January 1, 1970 to Generalized time
**/
DxError_t 
TLK_SCLK_ServiceClkRead(                    TLK_SCLK_ServiceClockDescriptor_t   *serviceDescr_ptr,
                                            TLK_SCLK_ApplicationId_t           *appId_ptr,
                                            TLK_SCLK_ServiceClockTime_t         *currTime_ptr,
                                            TLK_SCLK_TimeFormat_t                timeFormat,
                                            TLK_SCLK_ReadStatus_t               *status_ptr,
											DxUint8_t                           *serviceClkPwd,
											DxUint32_t                          serviceClkPwdLen,
                                            DxUint8_t                           *workspace_ptr,
                                            DxUint32_t                           workspaceSizeInBytes);

/*TLK_SCLK_MCServiceClkRevalidate*/
/**
* @brief     The purpose of this function is to verify and revalidate the service secured clock descriptor.
*
* @param serviceDescr_ptr           [in/out] - Pointer to the service clock descriptor to be verified.
* @param appId_ptr                  [in]     - Pointer to the application ID.
* @param expectedTime_ptr           [in]     - Pointer to the expected time for revalidation.
* @param timeFormat                 [in]     - Format in which time can be represented
* @param workspace_ptr              [in]     - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes       [in]     - Length of the workspace in bytes. The workspace size user can get by calling TLK_SCLK_WorkspaceSizeGet() API.
*
* @return TLK_SCLC_RC_OK                                     - The operation completed successfully.
* @return TLK_SCLK_RC_ERROR_API_NVS_CONFIGURATION            - The NVS configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_RTC_CONFIGURATION            - The RTC configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_TOOLKIT_ALREADY_INITIATED    - The toolkit initialization is done already. 
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NULL_PTR           - NULL Pinter workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NOT_ALIGN          - Unaligned workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE                    - Insufficient workspace given by user
* @return TLK_SCLK_RC_ERROR_API_NULL_POINTER                 - Null pointer received a input
* @return TLK_SCLK_RC_ERROR_MC_MAC_CALCULATE_FAIL            - MAC calculation of service clock failed
* @return TLK_SCLK_RC_ERROR_NVS_WRITE_MC_DESC_FAIL           - Cannot write master clock descriptor to VNVS
* @return TLK_SCLK_RC_ERROR_NVS_ENCRYPT_MC_DESC_FAIL         - Cannot encrypt master clock descriptor
* @return TLK_SCLK_RC_ERROR_MC_SERVICE_CLK_INTEGRITY         - Service clock integrity compromised
* @return TLK_SCLK_RC_ERROR_API_APP_ID_INVALID               - Application Id given by user is differ than application id in descriptor
* @return TLK_SCLK_RC_ERROR_RTC_TIME_GET                     - Error in RTC time getting
* @return TLK_SCLK_RC_ERROR_RTC_TIME_CONVERT_FAIL            - Problem in converting the given time Generalized time to number of seconds since January 1, 1970. or conversion time that is represented as the number of seconds since January 1, 1970 to Generalized time
* @return TLK_SCLK_RC_ERROR_API_MAC_CALCULATE_FAIL           - MAC calculation failed (Integrity problem)
**/
DxError_t 
TLK_SCLK_ServiceClkRevalidate(              TLK_SCLK_ServiceClockDescriptor_t   *serviceDescr_ptr,
                                            TLK_SCLK_ApplicationId_t            *appId_ptr,
                                    const   TLK_SCLK_ServiceClockTime_t         *expectedTime_ptr,
                                            TLK_SCLK_TimeFormat_t                timeFormat,
											DxUint8_t                           *serviceClkPwd,
											DxUint32_t                          serviceClkPwdLen,
                                            DxUint8_t                           *workspace_ptr,
                                            DxUint32_t                           workspaceSizeInBytes);

/*TLK_SCLK_Terminate*/
/**
* @brief     The purpose of this function is to terminate the work with master clock.
*
* @param workspace_ptr              [in]     - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes       [in]     - Length of the workspace in bytes. The workspace size user can get by calling TLK_SCLK_WorkspaceSizeGet() API.
*
* @return TLK_SCLC_RC_OK                                     - The operation completed successfully.
* @return TLK_SCLK_RC_ERROR_API_NVS_CONFIGURATION            - The NVS configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_RTC_CONFIGURATION            - The RTC configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_TOOLKIT_ALREADY_INITIATED    - The toolkit initialization is done already. 
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NULL_PTR           - NULL Pinter workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE_NOT_ALIGN          - Unaligned workspace given by user
* @return TLK_SCLK_RC_ERROR_API_WORKSPACE                    - Insufficient workspace given by user
* @return TLK_SCLK_RC_ERROR_API_TLK_SCLK_INACTIVE            - The toolkit is not initialized yet. Need to initialize
* @return TLK_SCLK_RC_ERROR_NVS_TERM_SESSION_FAIL            - In case of SST supports, Problem in session deletion 
* @return TLK_SCLK_RC_ERROR_NVS_TERM_TRANSACTION_FAIL        - In case of SST supports, Problem in transaction ending
* @return TLK_SCLK_RC_ERROR_NVS_TERM_FILE_CLOSE_FAIL         - Problem in NVS file or external file closing
* @return TLK_SCLK_RC_ERROR_MC_SERVICE_CLK_INTEGRITY         - Service clock integrity compromised
**/
DxError_t
TLK_SCLK_Terminate(                         DxUint8_t                           *workspace_ptr,
                                            DxUint32_t                           workspaceSizeInBytes);

/*TLK_SCLK_WorkspaceSizeGet*/
/**
* @brief     The purpose of this function is to get to user the minimum size of workspace in bytes.
*            User will be responsible for allocation of workspace with this size.
*
* @param workspaceSizeInBytes_ptr   [out]     - Required size of the workspace in bytes
*
* @return TLK_SCLC_RC_OK                                     - The operation completed successfully.
* @return TLK_SCLK_RC_ERROR_API_NVS_CONFIGURATION            - The NVS configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_RTC_CONFIGURATION            - The RTC configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_NULL_POINTER                 - Null pointer received a input
**/
DxError_t
TLK_SCLK_WorkspaceSizeGet(                  DxUint32_t                          *workspaceSizeInBytes_ptr);

/*TLK_SCLK_ServiceClkAuthenticate*/
/**
* @brief     The purpose of this function is Authenticate service clock and get its cookie and offset
*
* @param serviceDescr_ptr           [in]     - Pointer to the service clock descriptor to be verified.
* @param appId_ptr                  [in]     - Pointer to the application ID.
* @param serviceClkPwd              [in]     - Pointer to the service clock address.
* @param serviceClkPwdLen           [in]     - Length of the password.
* @param serviceClkCookie           [out]    - Buffer for the cookie.
* @param serviceClkOffset_ptr       [out]    - buffer for the offset.
* @param workspace_ptr              [in]     - Pointer to the workspace. Note: The user is responsible for workspace allocation.
* @param workspaceSizeInBytes       [in]     - Length of the workspace in bytes. The workspace size user can get by calling TLK_SCLK_WorkspaceSizeGet() API.
* @return TLK_SCLC_RC_OK                                     - The operation completed successfully.
* @return TLK_SCLK_RC_ERROR_API_NVS_CONFIGURATION            - The NVS configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_RTC_CONFIGURATION            - The RTC configuration in config file was defined wrong. User need to check configuration in config file.
* @return TLK_SCLK_RC_ERROR_API_NULL_POINTER                 - Null pointer received a input
**/
DxError_t  TLK_SCLK_ServiceClkAuthenticate( TLK_SCLK_ServiceClockDescriptor_t       *serviceDescr_ptr,
										   TLK_SCLK_ApplicationId_t                *appId_ptr,
										   DxUint8_t                               *serviceClkPwd,
										   DxUint32_t                              serviceClkPwdLen,
										   TLK_SCLK_Cookie_t						serviceClkCookie,
										   DxUint32_t                              *serviceClkOffset_ptr,
										   DxUint8_t                               *workspace_ptr,
										   DxUint32_t                               workspaceSizeInBytes);

#ifdef __cplusplus
}
#endif /*extern "C"*/

#endif /*_TLK_SCLK_API_H_*/
